unit PictureGrid;

interface

uses
  Windows,Graphics,Classes,Controls;

type
  TPicturePointer = ^TPicture;

  TPictureData = class(TPersistent)
  private
    FHeight:Integer;
    FWidth:Integer;
    FRows:Integer;
    FCols:Integer;
    FOnChange: TNotifyEvent;
  protected
    constructor Create;// override;
    procedure SetHeight(Value:Integer);
    procedure SetWidth(Value:Integer);
    procedure SetRows(Value:Integer);
    procedure SetCols(Value:Integer);
  public
    procedure Assign(Value: TPictureData);
    procedure Changed;
  published
    property Height:Integer Read FHeight Write SetHeight;
    property Width:Integer Read FWidth Write SetWidth;
    property Rows:Integer Read FRows Write SetRows;
    property Cols:Integer Read FCols Write SetCols;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
  end;

  TPictureGrid = class(TGraphicControl)
  private
    FPictArr: Array of Array of TPicturePointer;
    FPictureData:TPictureData;
    FAutoSize:Boolean;
  protected
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function GetPict(X, Y: Integer):TPicturePointer;
    procedure SetPict(X, Y: Integer; Value:TPicturePointer);
    function getPictData:TPictureData;
    procedure SetPictData(Value:TPictureData);
    procedure UpdatePictData(Sender: TObject);
    procedure SetAutoSize(Value:Boolean);
  public
    PictureTemplate: Array of TPicture;
    StartPos:TPoint;
    property Picture[X, Y: Integer]:TPicturePointer read GetPict write SetPict;
    procedure SetPictureTemplateSize(NewLength:Integer);
    procedure Paint; override;
    property canvas;
  published
    property Pictures:TPictureData Read FPictureData Write SetPictData;
    property AutoSize:Boolean Read FAutoSize Write SetAutoSize;
    property OnClick;
    property OnDblClick;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Additional', [TPictureGrid]);
end;

constructor TPictureData.Create;
begin
  FHeight := 32;
  FWidth := 32;
  FRows := 10;
  FCols := 10;
end;

procedure TPictureData.Assign(Value: TPictureData);
begin
  Height := Value.Height;
  Width := Value.Width;
  Rows := Value.Rows;
  Cols := Value.Cols;
  FOnChange(Self);
end;

procedure TPictureData.Changed;
begin
  FOnChange(Self);
end;

procedure TPictureData.SetHeight(Value:Integer);
begin
  FHeight := Value;
  Changed;
end;

procedure TPictureData.SetWidth(Value:Integer);
begin
  FWidth := Value;
  Changed;
end;

procedure TPictureData.SetRows(Value:Integer);
begin
  FRows := Value;
  Changed;
end;

procedure TPictureData.SetCols(Value:Integer);
begin
  FCols := Value;
  Changed;
end;

// TPictureGrid //

constructor TPictureGrid.Create(AOwner: TComponent);
var
  Numx,Numy:integer;
begin
  inherited Create(AOwner);
  FPictureData := TPictureData.Create;
  FPictureData.FOnChange := UpdatePictData;
  SetLength(FPictArr,FPictureData.Cols,FPictureData.Rows);
  AutoSize := True;
  Canvas.Pen.Style := psDot;
end;

destructor TPictureGrid.Destroy;
begin
  inherited Destroy;
  Setlength(FPictArr,0);
  Finalize(FPictArr);
  Setlength(PictureTemplate,0);
  Finalize(PictureTemplate);
end;

procedure TPictureGrid.Paint;
var
  Numx,Numy:integer;
  theRect:TRect;
begin
  with Canvas do begin
    AutoSize := FAutoSize; //If True, it will the Fix Size
    inherited Paint;
    for numx := 0 to FPictureData.Cols-1 do
      for numy := 0 to FPictureData.Rows-1 do begin
        theRect := Rect(FPictureData.Width*Numx+StartPos.x,FPictureData.Height*Numy+StartPos.y,FPictureData.Width*(Numx+1)+StartPos.x,FPictureData.Height*(Numy+1)+StartPos.y);
        If FPictArr[Numx,NumY] = nil then //Draw a Box
          Rectangle(theRect.Left,theRect.Top,theRect.Right+1,theRect.Bottom+1)
        else //Draw the Picture
          StretchDraw(theRect, FPictArr[Numx,Numy]^.graphic);
      end;
    FillRect(Rect(-Width*2,-Height*2,StartPos.x,Width*2));
    FillRect(Rect(-Width*2,-Height*2,Height*2, StartPos.y));
    FillRect(Rect(StartPos.x+FPictureData.Width*Numx,-Width*2,Width*2,Height*2));
    FillRect(Rect(-Height*2, StartPos.y+FPictureData.Height*Numy,Width*2,Height*2));
  end;
end;

function TPictureGrid.GetPict(X, Y: Integer):TPicturePointer;
begin
  Result := FPictArr[X,Y];
end;

procedure TPictureGrid.SetPict(X, Y: Integer; Value:TPicturePointer);
begin
  FPictArr[X,Y] := Value;
  //Paint;
end;

function TPictureGrid.getPictData:TPictureData;
begin
  SetLength(FPictArr,FPictureData.FCols,FPictureData.FRows);
end;

procedure TPictureGrid.SetPictData(Value:TPictureData);
begin
  FPictureData.Assign(Value);
  SetLength(FPictArr,FPictureData.FCols,FPictureData.FRows);
  Invalidate;
end;

procedure TPictureGrid.UpdatePictData(Sender: TObject);
begin
  SetLength(FPictArr,FPictureData.FCols,FPictureData.FRows);
  //AutoSize := FAutoSize;
  Invalidate;
end;

procedure TPictureGrid.SetPictureTemplateSize(NewLength:Integer);
begin
  SetLength(PictureTemplate, NewLength);
  Paint;
end;

procedure TPictureGrid.SetAutoSize(Value:Boolean);
begin
  FAutoSize := Value;
  If FAutoSize Then begin
    width := FPictureData.Width*FPictureData.Cols;
    height := FPictureData.Height*FPictureData.Rows;
  end;
end;

end.

